from django.contrib.auth import login, logout
from django.contrib.auth.forms import AuthenticationForm
from django.contrib.auth.models import User
from django.contrib.auth.mixins import LoginRequiredMixin
from django.contrib import messages
from django.shortcuts import redirect
from django.urls import reverse_lazy
from django.views.generic import (
    FormView, RedirectView, TemplateView,
    ListView, CreateView, UpdateView, DeleteView
)

# -------------------- Login & Logout --------------------
class CustomLoginView(FormView):
    template_name = 'registration/login.html'
    form_class = AuthenticationForm
    success_url = reverse_lazy('accounts:dashboard')

    def form_valid(self, form):
        user = form.get_user()
        login(self.request, user)
        messages.success(self.request, f"✅ Welcome back, {user.username}!")
        return super().form_valid(form)

    def form_invalid(self, form):
        messages.error(self.request, '❌ Invalid username or password.')
        return super().form_invalid(form)

class CustomLogoutView(RedirectView):
    pattern_name = 'accounts:login'

    def get(self, request, *args, **kwargs):
        logout(request)
        messages.success(request, "👋 You have been logged out.")
        return super().get(request, *args, **kwargs)


# -------------------- Dashboard --------------------
class DashboardView(LoginRequiredMixin, TemplateView):
    template_name = 'dashboard/index.html'

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['user_count'] = User.objects.count()
        context['active_users'] = User.objects.filter(is_active=True).count()
        context['staff_users'] = User.objects.filter(is_staff=True).count()
        return context


# -------------------- User Management --------------------
class UserListView(LoginRequiredMixin, ListView):
    model = User
    template_name = 'registration/user_list.html'
    context_object_name = 'users'


class UserCreateView(LoginRequiredMixin, CreateView):
    model = User
    fields = ['username', 'email', 'is_active', 'is_staff','is_superuser']
    template_name = 'registration/user_form.html'
    success_url = reverse_lazy('accounts:user_list')

    def form_valid(self, form):
        messages.success(self.request, "✅ User created successfully!")
        return super().form_valid(form)


class UserUpdateView(LoginRequiredMixin, UpdateView):
    model = User
    fields = ['username', 'email', 'is_active', 'is_staff','is_superuser']
    template_name = 'registration/user_form.html'
    success_url = reverse_lazy('accounts:user_list')

    def form_valid(self, form):
        messages.success(self.request, "✏️ User updated successfully!")
        return super().form_valid(form)


class UserDeleteView(LoginRequiredMixin, DeleteView):
    model = User
    template_name = 'registration/user_confirm_delete.html'
    success_url = reverse_lazy('accounts:user_list')

    def delete(self, request, *args, **kwargs):
        messages.warning(request, "🗑️ User deleted successfully!")
        return super().delete(request, *args, **kwargs)
