from django.contrib import messages
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from django.views.generic import ListView, CreateView, UpdateView, DeleteView

from enquiry.models import Enquiry, CareerEnquiry # Adjust path if needed
from .forms import EnquiryForm  # Optional: If using a custom form

# -------------------- Enquiry Management --------------------

from django.views.generic import ListView
from django.db.models import Q
from .models import Enquiry
from helpers import paginate_queryset

class EnquiryListView(LoginRequiredMixin, ListView):
    model = Enquiry
    template_name = 'enquiry_list.html'
    context_object_name = 'enquiries'

    def get_queryset(self):
        queryset = super().get_queryset().order_by("-created_at")
        request = self.request

        name = request.GET.get("name")
        email = request.GET.get("email")
        contact = request.GET.get("contact")
        city = request.GET.get("city")

        if name:
            queryset = queryset.filter(name__icontains=name)
        if email:
            queryset = queryset.filter(email__icontains=email)
        if contact:
            queryset = queryset.filter(contact__icontains=contact)
        if city:
            queryset = queryset.filter(city__icontains=city)

        return queryset

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        queryset = self.get_queryset()

        paginated_qs, page_obj = paginate_queryset(self.request, queryset)
        context["enquiries"] = paginated_qs
        context["page_obj"] = page_obj
        return context


class EnquiryCreateView(LoginRequiredMixin, CreateView):
    model = Enquiry
    fields = ['name', 'email', 'contact', 'city', 'message']
    template_name = 'enquiry/enquiry_form.html'
    success_url = reverse_lazy('enquiry:enquiry_list')

    def form_valid(self, form):
        messages.success(self.request, "✅ Enquiry submitted successfully!")
        return super().form_valid(form)


class EnquiryUpdateView(LoginRequiredMixin, UpdateView):
    model = Enquiry
    fields = ['name', 'email', 'contact', 'city', 'message']
    template_name = 'enquiry/enquiry_form.html'
    success_url = reverse_lazy('enquiry:enquiry_list')

    def form_valid(self, form):
        messages.success(self.request, "✏️ Enquiry updated successfully!")
        return super().form_valid(form)


class EnquiryDeleteView(LoginRequiredMixin, DeleteView):
    model = Enquiry
    template_name = 'enquiry_confirm_delete.html'
    success_url = reverse_lazy('enquiry:enquiry_list')

    def delete(self, request, *args, **kwargs):
        messages.warning(request, "🗑️ Enquiry deleted successfully!")
        return super().delete(request, *args, **kwargs)




#Career 

class CareerListView(LoginRequiredMixin, ListView):
    model = CareerEnquiry
    template_name = 'career/career_list.html'
    context_object_name = 'career_enquires'

    def get_queryset(self):
        queryset = super().get_queryset().order_by("-created_at")
        request = self.request

        name = request.GET.get("name")
        email = request.GET.get("email")
        contact = request.GET.get("contact")
       

        if name:
            queryset = queryset.filter(name__icontains=name)
        if email:
            queryset = queryset.filter(email__icontains=email)
        if contact:
            queryset = queryset.filter(contact__icontains=contact)
      

        return queryset

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        queryset = self.get_queryset()

        paginated_qs, page_obj = paginate_queryset(self.request, queryset)
        context["career_enquires"] = paginated_qs
        context["page_obj"] = page_obj
        return context


class CareerCreateView(LoginRequiredMixin, CreateView):
    model = CareerEnquiry
    fields = ['name', 'email', 'contact', 'city', 'message']
    template_name = 'enquiry/enquiry_form.html'
    success_url = reverse_lazy('enquiry:enquiry_list')

    def form_valid(self, form):
        messages.success(self.request, "✅ Enquiry submitted successfully!")
        return super().form_valid(form)


class CareerUpdateView(LoginRequiredMixin, UpdateView):
    model = CareerEnquiry
    fields = ['name', 'email', 'contact', 'city', 'message']
    template_name = 'enquiry/enquiry_form.html'
    success_url = reverse_lazy('enquiry:enquiry_list')

    def form_valid(self, form):
        messages.success(self.request, "✏️ Enquiry updated successfully!")
        return super().form_valid(form)


class CareerDeleteView(LoginRequiredMixin, DeleteView):
    model = CareerEnquiry
    template_name = 'enquiry_confirm_delete.html'
    success_url = reverse_lazy('enquiry:enquiry_list')

    def delete(self, request, *args, **kwargs):
        messages.warning(request, "🗑️ Enquiry deleted successfully!")
        return super().delete(request, *args, **kwargs)
