# helpers.py

def apply_form_styling_and_validations(form, required=True, exclude=[]):
    """
    Apply consistent styling and HTML5 validations to all fields of a Django form.

    Parameters:
        - form: The Django form instance.
        - required: If True, makes all fields required (unless excluded).
        - exclude: List of field names to exclude from styling/validation.
    """
    for name, field in form.fields.items():
        if name in exclude:
            continue

        # Apply required flag
        if required and not field.required:
            field.required = True

        # Apply form-control class
        existing_classes = field.widget.attrs.get('class', '')
        field.widget.attrs['class'] = f'{existing_classes} form-control'.strip()

        # Placeholder (optional fallback)
        if 'placeholder' not in field.widget.attrs:
            field.widget.attrs['placeholder'] = field.label

        # HTML5 validation
        field.widget.attrs['oninvalid'] = (
            f"this.setCustomValidity('Please enter a valid {field.label}')"
        )
        field.widget.attrs['oninput'] = "this.setCustomValidity('')"

        # Type-specific attributes
        if hasattr(field, 'max_length') and field.max_length:
            field.widget.attrs['maxlength'] = field.max_length

        if field.__class__.__name__ in ['EmailField']:
            field.widget.attrs['type'] = 'email'
        elif field.__class__.__name__ in ['URLField']:
            field.widget.attrs['type'] = 'url'
        elif field.__class__.__name__ in ['IntegerField']:
            field.widget.attrs['type'] = 'number'
        elif field.__class__.__name__ in ['BooleanField']:
            field.widget.attrs['class'] = 'form-check-input'

        # Date fields
        if field.__class__.__name__ in ['DateField', 'DateTimeField']:
            field.widget.attrs['type'] = 'date'  # or 'datetime-local' if needed

from django.db import models
from django.utils import timezone

class SoftDeleteModel(models.Model):
    deleted_at = models.DateTimeField(null=True, blank=True)

    class Meta:
        abstract = True

    def delete(self, using=None, keep_parents=False):
        self.deleted_at = timezone.now()
        self.save()

    def hard_delete(self, using=None, keep_parents=False):
        super().delete(using=using, keep_parents=keep_parents)

    @property
    def is_deleted(self):
        return self.deleted_at is not None


#Paginator
from django.core.paginator import Paginator, EmptyPage, PageNotAnInteger

PER_PAGE = 10  # 👈 Define your default per-page value here

def paginate_queryset(request, queryset, per_page=PER_PAGE):
    paginator = Paginator(queryset, per_page)
    page = request.GET.get('page')
    try:
        paginated_qs = paginator.page(page)
    except PageNotAnInteger:
        paginated_qs = paginator.page(1)
    except EmptyPage:
        paginated_qs = paginator.page(paginator.num_pages)
    return paginated_qs, paginated_qs


