from django.contrib import messages
from django.contrib.auth.mixins import LoginRequiredMixin
from django.urls import reverse_lazy
from django.views.generic import ListView, CreateView, UpdateView, DeleteView

from master.models import City  # Adjust path if needed


# -------------------- Enquiry Management --------------------

from django.views.generic import ListView
from django.db.models import Q

from helpers import paginate_queryset

class CityListView(LoginRequiredMixin, ListView):
    model = City
    template_name = 'city/city_list.html'
    context_object_name = 'cities'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        queryset = self.get_queryset()

        paginated_qs, page_obj = paginate_queryset(self.request, queryset)
        context['cities'] = paginated_qs
        context['page_obj'] = page_obj
        return context


class CityCreateView(LoginRequiredMixin, CreateView):
    model = City
    fields = ['name']
    template_name = 'city/city_form.html'
    success_url = reverse_lazy('city:city_list')

    def form_valid(self, form):
        messages.success(self.request, "✅ City submitted successfully!")
        return super().form_valid(form)


class CityUpdateView(LoginRequiredMixin, UpdateView):
    model = City
    fields = ['name']
    template_name = 'city/city_form.html'
    success_url = reverse_lazy('city:city_list')

    def form_valid(self, form):
        messages.success(self.request, "✏️ City updated successfully!")
        return super().form_valid(form)


class CityDeleteView(LoginRequiredMixin, DeleteView):
    model = City
    template_name = 'city/city_confirm_delete.html'
    success_url = reverse_lazy('city:city_list')


    def delete(self, request, *args, **kwargs):
        messages.warning(request, "🗑️ City deleted successfully!")
        return super().delete(request, *args, **kwargs)
